<?php
// pages/settings.php
require_once __DIR__ . '/../config/auth_check.php';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Settings - AdMob Analytics</title>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <?php include __DIR__ . '/../includes/sidebar.php'; ?>
    
    <div class="main-content">
        <?php include __DIR__ . '/../includes/header.php'; ?>
        
        <div class="content-wrapper">
            <div class="page-header">
                <h1>Settings</h1>
            </div>

            <!-- API Configuration -->
            <div class="card">
                <div class="card-header">
                    <h2><i class="fas fa-cog"></i> API Configuration</h2>
                </div>
                <div class="card-body">
                    <div class="settings-section">
                        <div class="form-group">
                            <label>Google OAuth Status</label>
                            <div class="status-badge" id="oauth-status">
                                <i class="fas fa-spinner fa-spin"></i> Checking...
                            </div>
                        </div>

                        <div class="form-group">
                            <label>Client ID</label>
                            <input type="text" class="form-control" value="<?php echo substr(GOOGLE_CLIENT_ID, 0, 20); ?>..." readonly>
                        </div>

                        <div class="form-group">
                            <label>Redirect URI</label>
                            <input type="text" class="form-control" value="<?php echo GOOGLE_REDIRECT_URI; ?>" readonly>
                        </div>

                        <button class="btn btn-secondary" onclick="testConnection()">
                            <i class="fas fa-plug"></i> Test Connection
                        </button>
                        <button class="btn btn-danger" onclick="revokeAccess()">
                            <i class="fas fa-sign-out-alt"></i> Revoke Access
                        </button>
                    </div>
                </div>
            </div>

            <!-- Data Sync Settings -->
            <div class="card">
                <div class="card-header">
                    <h2><i class="fas fa-sync-alt"></i> Data Synchronization</h2>
                </div>
                <div class="card-body">
                    <div class="settings-section">
                        <div class="form-group">
                            <label>Auto Sync</label>
                            <div class="toggle-switch">
                                <input type="checkbox" id="auto-sync" checked>
                                <label for="auto-sync"></label>
                            </div>
                            <small>Automatically sync data from AdMob every day</small>
                        </div>

                        <div class="form-group">
                            <label>Sync Frequency</label>
                            <select class="form-control" id="sync-frequency">
                                <option value="hourly">Every Hour</option>
                                <option value="daily" selected>Daily</option>
                                <option value="weekly">Weekly</option>
                            </select>
                        </div>

                        <div class="form-group">
                            <label>Last Sync</label>
                            <div id="last-sync-time">Never</div>
                        </div>

                        <button class="btn btn-primary" onclick="syncNow()">
                            <i class="fas fa-sync-alt"></i> Sync Now
                        </button>
                    </div>
                </div>
            </div>

            <!-- Report Settings -->
            <div class="card">
                <div class="card-header">
                    <h2><i class="fas fa-file-alt"></i> Report Settings</h2>
                </div>
                <div class="card-body">
                    <div class="settings-section">
                        <div class="form-group">
                            <label>Default Date Range</label>
                            <select class="form-control" id="default-date-range">
                                <option value="today">Today</option>
                                <option value="yesterday">Yesterday</option>
                                <option value="last_7_days" selected>Last 7 Days</option>
                                <option value="last_30_days">Last 30 Days</option>
                                <option value="this_month">This Month</option>
                            </select>
                        </div>

                        <div class="form-group">
                            <label>Default Dimension</label>
                            <select class="form-control" id="default-dimension">
                                <option value="date" selected>Date</option>
                                <option value="app">App</option>
                                <option value="ad_unit">Ad Unit</option>
                                <option value="country">Country</option>
                            </select>
                        </div>

                        <div class="form-group">
                            <label>Default Export Format</label>
                            <select class="form-control" id="default-export-format">
                                <option value="csv" selected>CSV</option>
                                <option value="json">JSON</option>
                                <option value="xlsx">Excel (XLSX)</option>
                            </select>
                        </div>

                        <div class="form-group">
                            <label>Cache Reports</label>
                            <div class="toggle-switch">
                                <input type="checkbox" id="cache-reports" checked>
                                <label for="cache-reports"></label>
                            </div>
                            <small>Cache report results for faster loading</small>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Display Settings -->
            <div class="card">
                <div class="card-header">
                    <h2><i class="fas fa-paint-brush"></i> Display Settings</h2>
                </div>
                <div class="card-body">
                    <div class="settings-section">
                        <div class="form-group">
                            <label>Currency Symbol</label>
                            <select class="form-control" id="currency-symbol">
                                <option value="USD" selected>$ (USD)</option>
                                <option value="EUR">€ (EUR)</option>
                                <option value="GBP">£ (GBP)</option>
                                <option value="PKR">Rs. (PKR)</option>
                            </select>
                        </div>

                        <div class="form-group">
                            <label>Number Format</label>
                            <select class="form-control" id="number-format">
                                <option value="en-US" selected>1,234.56 (US)</option>
                                <option value="en-GB">1,234.56 (UK)</option>
                                <option value="de-DE">1.234,56 (DE)</option>
                            </select>
                        </div>

                        <div class="form-group">
                            <label>Date Format</label>
                            <select class="form-control" id="date-format">
                                <option value="MM/DD/YYYY" selected>MM/DD/YYYY</option>
                                <option value="DD/MM/YYYY">DD/MM/YYYY</option>
                                <option value="YYYY-MM-DD">YYYY-MM-DD</option>
                            </select>
                        </div>

                        <div class="form-group">
                            <label>Items Per Page</label>
                            <select class="form-control" id="items-per-page">
                                <option value="10">10</option>
                                <option value="25" selected>25</option>
                                <option value="50">50</option>
                                <option value="100">100</option>
                            </select>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Database Settings -->
            <div class="card">
                <div class="card-header">
                    <h2><i class="fas fa-database"></i> Database Management</h2>
                </div>
                <div class="card-body">
                    <div class="settings-section">
                        <div class="form-group">
                            <label>Database Status</label>
                            <div class="status-badge" id="db-status">
                                <i class="fas fa-check-circle" style="color: #4CAF50;"></i> Connected
                            </div>
                        </div>

                        <div class="form-group">
                            <label>Clear Cache</label>
                            <button class="btn btn-secondary" onclick="clearCache()">
                                <i class="fas fa-trash"></i> Clear All Cached Reports
                            </button>
                            <small>Remove all cached report data to free up space</small>
                        </div>

                        <div class="form-group">
                            <label>Data Retention</label>
                            <select class="form-control" id="data-retention">
                                <option value="30">30 days</option>
                                <option value="90" selected>90 days</option>
                                <option value="180">180 days</option>
                                <option value="365">1 year</option>
                                <option value="0">Keep all data</option>
                            </select>
                            <small>Automatically delete data older than selected period</small>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Save Settings -->
            <div class="settings-actions">
                <button class="btn btn-primary btn-lg" onclick="saveSettings()">
                    <i class="fas fa-save"></i> Save All Settings
                </button>
                <button class="btn btn-secondary btn-lg" onclick="resetSettings()">
                    <i class="fas fa-undo"></i> Reset to Defaults
                </button>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            checkOAuthStatus();
            loadSettings();
        });

        async function checkOAuthStatus() {
            const statusElement = document.getElementById('oauth-status');
            
            try {
                const response = await fetch('/api/account.php?action=info');
                const data = await response.json();

                if (data.success) {
                    statusElement.innerHTML = '<i class="fas fa-check-circle" style="color: #4CAF50;"></i> Connected';
                } else {
                    statusElement.innerHTML = '<i class="fas fa-times-circle" style="color: #f44336;"></i> Not Connected';
                }
            } catch (error) {
                statusElement.innerHTML = '<i class="fas fa-exclamation-triangle" style="color: #FF9800;"></i> Unknown';
            }
        }

        async function testConnection() {
            const btn = event.target;
            btn.disabled = true;
            btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Testing...';

            try {
                const response = await fetch('/api/account.php?action=info');
                const data = await response.json();

                if (data.success) {
                    alert('Connection successful! ✓');
                } else {
                    alert('Connection failed: ' + data.error);
                }
            } catch (error) {
                alert('Connection test failed: ' + error.message);
            } finally {
                btn.disabled = false;
                btn.innerHTML = '<i class="fas fa-plug"></i> Test Connection';
            }
        }

        function revokeAccess() {
            if (confirm('Are you sure you want to revoke access? You will need to re-authenticate.')) {
                window.location.href = '/logout.php';
            }
        }

        async function syncNow() {
            const btn = event.target;
            btn.disabled = true;
            btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Syncing...';

            try {
                // Sync apps
                await fetch('/api/apps.php?action=sync', { method: 'POST' });
                // Sync ad units
                await fetch('/api/ad_units.php?action=sync', { method: 'POST' });
                // Sync account
                await fetch('/api/account.php?action=sync');

                alert('Data synced successfully! ✓');
                document.getElementById('last-sync-time').textContent = new Date().toLocaleString();
            } catch (error) {
                alert('Sync failed: ' + error.message);
            } finally {
                btn.disabled = false;
                btn.innerHTML = '<i class="fas fa-sync-alt"></i> Sync Now';
            }
        }

        async function clearCache() {
            if (confirm('Are you sure you want to clear all cached reports?')) {
                try {
                    const response = await fetch('/api/reports.php?action=clear_cache', {
                        method: 'POST'
                    });
                    const data = await response.json();

                    if (data.success) {
                        alert('Cache cleared successfully! ✓');
                    } else {
                        alert('Failed to clear cache');
                    }
                } catch (error) {
                    alert('Error: ' + error.message);
                }
            }
        }

        function loadSettings() {
            // Load settings from localStorage
            const settings = JSON.parse(localStorage.getItem('appSettings') || '{}');

            if (settings.autoSync !== undefined) {
                document.getElementById('auto-sync').checked = settings.autoSync;
            }
            if (settings.syncFrequency) {
                document.getElementById('sync-frequency').value = settings.syncFrequency;
            }
            if (settings.defaultDateRange) {
                document.getElementById('default-date-range').value = settings.defaultDateRange;
            }
            if (settings.defaultDimension) {
                document.getElementById('default-dimension').value = settings.defaultDimension;
            }
            if (settings.defaultExportFormat) {
                document.getElementById('default-export-format').value = settings.defaultExportFormat;
            }
            if (settings.cacheReports !== undefined) {
                document.getElementById('cache-reports').checked = settings.cacheReports;
            }
            if (settings.currencySymbol) {
                document.getElementById('currency-symbol').value = settings.currencySymbol;
            }
            if (settings.numberFormat) {
                document.getElementById('number-format').value = settings.numberFormat;
            }
            if (settings.dateFormat) {
                document.getElementById('date-format').value = settings.dateFormat;
            }
            if (settings.itemsPerPage) {
                document.getElementById('items-per-page').value = settings.itemsPerPage;
            }
            if (settings.dataRetention) {
                document.getElementById('data-retention').value = settings.dataRetention;
            }
        }

        function saveSettings() {
            const settings = {
                autoSync: document.getElementById('auto-sync').checked,
                syncFrequency: document.getElementById('sync-frequency').value,
                defaultDateRange: document.getElementById('default-date-range').value,
                defaultDimension: document.getElementById('default-dimension').value,
                defaultExportFormat: document.getElementById('default-export-format').value,
                cacheReports: document.getElementById('cache-reports').checked,
                currencySymbol: document.getElementById('currency-symbol').value,
                numberFormat: document.getElementById('number-format').value,
                dateFormat: document.getElementById('date-format').value,
                itemsPerPage: document.getElementById('items-per-page').value,
                dataRetention: document.getElementById('data-retention').value
            };

            localStorage.setItem('appSettings', JSON.stringify(settings));
            alert('Settings saved successfully! ✓');
        }

        function resetSettings() {
            if (confirm('Are you sure you want to reset all settings to defaults?')) {
                localStorage.removeItem('appSettings');
                location.reload();
            }
        }
    </script>
</body>
</html>