<?php
// pages/account.php
require_once __DIR__ . '/../config/auth_check.php';
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Account - AdMob Analytics</title>
    <link rel="stylesheet" href="/assets/css/style.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body>
    <?php include __DIR__ . '/../includes/sidebar.php'; ?>
    
    <div class="main-content">
        <?php include __DIR__ . '/../includes/header.php'; ?>
        
        <div class="content-wrapper">
            <div class="page-header">
                <h1>Account Information</h1>
                <div class="page-actions">
                    <button class="btn btn-primary" onclick="syncAccount()">
                        <i class="fas fa-sync-alt"></i> Sync Account
                    </button>
                </div>
            </div>

            <!-- Account Overview -->
            <div class="card">
                <div class="card-header">
                    <h2><i class="fas fa-user-circle"></i> Account Details</h2>
                </div>
                <div class="card-body">
                    <div id="account-details" class="account-details">
                        <div class="loading-state">
                            <i class="fas fa-spinner fa-spin"></i> Loading account information...
                        </div>
                    </div>
                </div>
            </div>

            <!-- Account Statistics -->
            <div class="dashboard-row">
                <div class="card">
                    <div class="card-header">
                        <h2><i class="fas fa-chart-pie"></i> Account Summary</h2>
                    </div>
                    <div class="card-body">
                        <div id="account-summary" class="summary-grid">
                            <div class="loading-state">
                                <i class="fas fa-spinner fa-spin"></i> Loading...
                            </div>
                        </div>
                    </div>
                </div>

                <div class="card">
                    <div class="card-header">
                        <h2><i class="fas fa-history"></i> Recent Activity</h2>
                    </div>
                    <div class="card-body">
                        <div id="recent-activity">
                            <div class="loading-state">
                                <i class="fas fa-spinner fa-spin"></i> Loading...
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Publisher Information -->
            <div class="card">
                <div class="card-header">
                    <h2><i class="fas fa-info-circle"></i> Publisher Information</h2>
                </div>
                <div class="card-body">
                    <div id="publisher-info" class="app-details">
                        <div class="loading-state">
                            <i class="fas fa-spinner fa-spin"></i> Loading...
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            loadAccountInfo();
            loadAccountStats();
        });

        async function loadAccountInfo() {
            try {
                const response = await fetch('/api/account.php?action=info');
                const data = await response.json();

                if (data.success) {
                    displayAccountDetails(data.data);
                    displayPublisherInfo(data.data);
                } else {
                    showError('account-details', data.error);
                }
            } catch (error) {
                console.error('Error loading account info:', error);
                showError('account-details', 'Failed to load account information');
            }
        }

        function displayAccountDetails(account) {
            const container = document.getElementById('account-details');
            
            container.innerHTML = `
                <div class="detail-row">
                    <label><i class="fas fa-id-badge"></i> Publisher ID:</label>
                    <span><code>${account.publisher_id || 'N/A'}</code></span>
                </div>
                <div class="detail-row">
                    <label><i class="fas fa-building"></i> Account Name:</label>
                    <span>${account.account_name || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <label><i class="fas fa-dollar-sign"></i> Currency:</label>
                    <span>${account.currency_code || 'USD'}</span>
                </div>
                <div class="detail-row">
                    <label><i class="fas fa-clock"></i> Timezone:</label>
                    <span>${account.timezone || 'UTC'}</span>
                </div>
                <div class="detail-row">
                    <label><i class="fas fa-sync"></i> Last Updated:</label>
                    <span>${new Date(account.last_updated).toLocaleString()}</span>
                </div>
            `;
        }

        function displayPublisherInfo(account) {
            const container = document.getElementById('publisher-info');
            const accountData = account.account_data || {};
            
            container.innerHTML = `
                <div class="detail-row">
                    <label>Display Name:</label>
                    <span>${accountData.displayName || account.account_name || 'N/A'}</span>
                </div>
                <div class="detail-row">
                    <label>Publisher ID:</label>
                    <span><code>${accountData.name || account.publisher_id || 'N/A'}</code></span>
                </div>
                <div class="detail-row">
                    <label>Currency Code:</label>
                    <span>${accountData.currencyCode || account.currency_code || 'USD'}</span>
                </div>
                <div class="detail-row">
                    <label>Report Timezone:</label>
                    <span>${accountData.reportingTimeZone || account.timezone || 'UTC'}</span>
                </div>
            `;
        }

        async function loadAccountStats() {
            try {
                const response = await fetch('/api/account.php?action=stats');
                const data = await response.json();

                if (data.success) {
                    displayAccountSummary(data.data);
                } else {
                    showError('account-summary', data.error);
                }
            } catch (error) {
                console.error('Error loading account stats:', error);
                showError('account-summary', 'Failed to load statistics');
            }

            loadRecentActivity();
        }

        function displayAccountSummary(stats) {
            const container = document.getElementById('account-summary');
            
            container.innerHTML = `
                <div class="summary-item">
                    <div class="summary-icon" style="background: #4CAF50;">
                        <i class="fas fa-mobile-alt"></i>
                    </div>
                    <div class="summary-content">
                        <h3>${stats.total_apps || 0}</h3>
                        <p>Total Apps</p>
                    </div>
                </div>
                <div class="summary-item">
                    <div class="summary-icon" style="background: #2196F3;">
                        <i class="fas fa-ad"></i>
                    </div>
                    <div class="summary-content">
                        <h3>${stats.total_ad_units || 0}</h3>
                        <p>Ad Units</p>
                    </div>
                </div>
            `;
        }

        async function loadRecentActivity() {
            const container = document.getElementById('recent-activity');
            
            // Mock recent activity - you can replace with actual API call
            container.innerHTML = `
                <div class="activity-list">
                    <div class="activity-item">
                        <div class="activity-icon">
                            <i class="fas fa-sync"></i>
                        </div>
                        <div class="activity-content">
                            <p>Account synced successfully</p>
                            <small>Just now</small>
                        </div>
                    </div>
                    <div class="activity-item">
                        <div class="activity-icon">
                            <i class="fas fa-mobile-alt"></i>
                        </div>
                        <div class="activity-content">
                            <p>Apps imported from AdMob</p>
                            <small>2 hours ago</small>
                        </div>
                    </div>
                    <div class="activity-item">
                        <div class="activity-icon">
                            <i class="fas fa-chart-line"></i>
                        </div>
                        <div class="activity-content">
                            <p>Reports generated</p>
                            <small>1 day ago</small>
                        </div>
                    </div>
                </div>
            `;
        }

        async function syncAccount() {
            const btn = event.target;
            btn.disabled = true;
            btn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Syncing...';

            try {
                const response = await fetch('/api/account.php?action=sync');
                const data = await response.json();

                if (data.success) {
                    alert('Account synced successfully!');
                    loadAccountInfo();
                    loadAccountStats();
                } else {
                    alert('Error: ' + data.error);
                }
            } catch (error) {
                console.error('Error syncing account:', error);
                alert('Failed to sync account');
            } finally {
                btn.disabled = false;
                btn.innerHTML = '<i class="fas fa-sync-alt"></i> Sync Account';
            }
        }

        function showError(containerId, message) {
            const container = document.getElementById(containerId);
            container.innerHTML = `<div class="error-state"><i class="fas fa-exclamation-triangle"></i><p>${message}</p></div>`;
        }
    </script>
</body>
</html>